import bisect 
import os
import pickle

from PIL import Image
import numpy as np
import torch
from torch.utils.data import Dataset, DataLoader
from torchvision import transforms, datasets
from typing import *

# list of all datasets
DATASETS = ["cifar10", "mnist"]

def get_dataset(dataset: str, split: str) -> Dataset:
    """Return the dataset as a PyTorch Dataset object"""
    if dataset == "cifar10":
        return _cifar10(split)

    elif dataset == "mnist":
        return _mnist(split)

def get_num_classes(dataset: str):
    """Return the number of classes in the dataset. """
    if dataset == "cifar10":
        return 10
    elif dataset == "mnist":
        return 10

def get_normalize_layer(dataset: str) -> torch.nn.Module:
    """Return the dataset's normalization layer"""
    if dataset == "cifar10":
        return NormalizeLayer(_CIFAR10_MEAN, _CIFAR10_STDDEV)
    elif dataset == "mnist":
        return NormalizeLayer(_MNIST_MEAN, _MNIST_STDDEV)

def get_input_center_layer(dataset: str) -> torch.nn.Module:
    """Return the dataset's Input Centering layer"""
    if dataset == "cifar10":
        return InputCenterLayer(_CIFAR10_MEAN)
    elif dataset == "mnist":
        return InputCenterLayer(_MNIST_MEAN)

_CIFAR10_MEAN = [0.4914, 0.4822, 0.4465]
_CIFAR10_STDDEV = [0.2023, 0.1994, 0.2010]

_MNIST_MEAN = [0.1306]
_MNIST_STDDEV = [0.3081]

def _cifar10(split: str) -> Dataset:
    dataset_path = os.path.join('datasets', 'cifar10')
    if split == "train":
        return datasets.CIFAR10(dataset_path, train=True, download=True, transform=transforms.Compose([
            transforms.RandomCrop(32, padding=4),
            transforms.RandomHorizontalFlip(),
            transforms.ToTensor()
        ]))
    elif split == "test":
        return datasets.CIFAR10(dataset_path, train=False, download=True, transform=transforms.ToTensor())

    else:
        raise Exception("Unknown split name.")

def _mnist(split: str) -> Dataset:
    dataset_path = os.path.join('datasets', 'mnist')
    if split == "train":
        return datasets.MNIST(dataset_path, train=True, download=True, transform=transforms.ToTensor())
    elif split == "test":
        return datasets.MNIST(dataset_path, train=False, download=True, transform=transforms.ToTensor())

    else:
        raise Exception("Unknown split name.")

class SigmaDataset(Dataset):
    def __init__(self, cifar_data, sigma_file, transform=None):
        f = open(sigma_file, "r")
        self.sigma_file = lines = f.readlines()
        self.cifar_data = cifar_data
        f.close()

    def __len__(self):
        return len(self.cifar_data)
 
    def __getitem__(self, index):
        data = self.cifar_data[index]
        line = self.sigma_file[index]
        line = line.split("\t")
        sigma = float(line[-1])

        return data[0], data[1], sigma

class NormalizeLayer(torch.nn.Module):
    """Standardize the channels of a batch of images by subtracting the dataset mean
      and dividing by the dataset standard deviation.

      In order to certify radii in original coordinates rather than standardized coordinates, we
      add the Gaussian noise _before_ standardizing, which is why we have standardization be the first
      layer of the classifier rather than as a part of preprocessing as is typical.
      """

    def __init__(self, means: List[float], sds: List[float]):
        """
        :param means: the channel means
        :param sds: the channel standard deviations
        """
        super(NormalizeLayer, self).__init__()
        self.means = torch.tensor(means).cuda()
        self.sds = torch.tensor(sds).cuda()

    def forward(self, input: torch.tensor):
        (batch_size, num_channels, height, width) = input.shape
        means = self.means.repeat((batch_size, height, width, 1)).permute(0, 3, 1, 2)
        sds = self.sds.repeat((batch_size, height, width, 1)).permute(0, 3, 1, 2)
        return (input - means)/sds


class InputCenterLayer(torch.nn.Module):
    """Centers the channels of a batch of images by subtracting the dataset mean.

      In order to certify radii in original coordinates rather than standardized coordinates, we
      add the Gaussian noise _before_ standardizing, which is why we have standardization be the first
      layer of the classifier rather than as a part of preprocessing as is typical.
      """

    def __init__(self, means: List[float]):
        """
        :param means: the channel means
        :param sds: the channel standard deviations
        """
        super(InputCenterLayer, self).__init__()
        self.means = torch.tensor(means).cuda()

    def forward(self, input: torch.tensor):
        (batch_size, num_channels, height, width) = input.shape
        means = self.means.repeat((batch_size, height, width, 1)).permute(0, 3, 1, 2)
        return input - means

